#include "App1.h"

App1::App1()
{
}

void App1::init(HINSTANCE hinstance, HWND hwnd, int screenWidth, int screenHeight, Input *in, bool VSYNC, bool FULL_SCREEN)
{
	// Call super/parent init function (required!)
	BaseApplication::init(hinstance, hwnd, screenWidth, screenHeight, in, VSYNC, FULL_SCREEN);

	//Load all our textures
	textureMgr->loadTexture(L"skybox", L"res/Textures/skybox.png");
	textureMgr->loadTexture(L"textureMap", L"res/Textures/terrainTextureMap.dds");
	textureMgr->loadTexture(L"grass", L"res/Textures/Grass.dds");
	textureMgr->loadTexture(L"rock", L"res/Textures/Rock.dds");
	textureMgr->loadTexture(L"dirt", L"res/Textures/Dirt.dds");
	textureMgr->loadTexture(L"sand", L"res/Textures/Sand.dds");
	textureMgr->loadTexture(L"billboardGrass", L"res/Textures/grassBillboard.png");
	textureMgr->loadTexture(L"billboardTree", L"res/Textures/treeBillboard.png");
	textureMgr->loadTexture(L"testDisplacement", L"res/Textures/terrainHeightMap.png");
	textureMgr->loadTexture(L"wood", L"res/Textures/wood.png");
	textureMgr->loadTexture(L"dudv", L"res/Textures/waterDUDV.png");
	textureMgr->loadTexture(L"waterNormal", L"res/Textures/waterNormal.png");
	textureMgr->loadTexture(L"hutTexture", L"res/Textures/Hut.dds");

	//Create all our models/geometry/meshes
	skybox = new Skybox(renderer->getDevice(), renderer->getDeviceContext());
	terrainPlane = new TessellationPlane(renderer->getDevice(), renderer->getDeviceContext());
	waterPlane = new PlaneMesh(renderer->getDevice(), renderer->getDeviceContext());
	refractionWoodCube = new CubeMesh(renderer->getDevice(), renderer->getDeviceContext());
	hutModel = new AModel(renderer->getDevice(), "res/Models/Hut.FBX");

	//Create shadow maps
	sunShadowMap = new ShadowMap(renderer->getDevice(), 8192, 8192);
	spotShadowMap = new ShadowMap(renderer->getDevice(), 8192, 8192);

	for (int i = 0; i < 6; i++)
	{
		hillShadowMaps[i] = new ShadowMap(renderer->getDevice(), 8192, 8192);
	}

	//Initialize shaders, geometry and lighting
	InitializeShaders(hwnd, screenWidth, screenHeight);
	GetGeometryPositions();
	InitializeLighting();
	SetLights();

	//Create all render textures
	reflectionRenderTexture = new RenderTexture(renderer->getDevice(), screenWidth, screenHeight, SCREEN_NEAR, SCREEN_DEPTH);
	refractionRenderTexture = new RenderTexture(renderer->getDevice(), screenWidth, screenHeight, SCREEN_NEAR, SCREEN_DEPTH);
	sceneRenderTexture = new RenderTexture(renderer->getDevice(), screenWidth, screenHeight, SCREEN_NEAR, SCREEN_DEPTH);
	motionBlurTexture = new RenderTexture(renderer->getDevice(), screenWidth, screenHeight, SCREEN_NEAR, SCREEN_DEPTH);
	glowTexture = new RenderTexture(renderer->getDevice(), screenWidth, screenHeight, SCREEN_NEAR, SCREEN_DEPTH);
	glowBlurTexture = new RenderTexture(renderer->getDevice(), screenWidth, screenHeight, SCREEN_NEAR, SCREEN_DEPTH);
	finalBloomTexture = new RenderTexture(renderer->getDevice(), screenWidth, screenHeight, SCREEN_NEAR, SCREEN_DEPTH);

	//Create all ortho meshes
	sceneOrtho = new OrthoMesh(renderer->getDevice(), renderer->getDeviceContext(), screenWidth, screenHeight);
	reflectionOrtho = new OrthoMesh(renderer->getDevice(), renderer->getDeviceContext(), screenWidth / 4.0f, screenHeight / 4.0f, -screenWidth / 2.7, screenHeight / 2.7);
	refractionOrtho = new OrthoMesh(renderer->getDevice(), renderer->getDeviceContext(), screenWidth / 4.0f, screenHeight / 4.0f, screenWidth / 2.7, screenHeight / 2.7);
	shadowOrtho = new OrthoMesh(renderer->getDevice(), renderer->getDeviceContext(), screenWidth / 4.0f, screenHeight / 4.0f, 0.0f, screenHeight / 2.7);

	//Initialize/set all default values
	skybox->SetPosition(XMFLOAT3(camera->getPosition().x - 0.5f, camera->getPosition().y - 0.5f, camera->getPosition().z + 0.5f));
	waterRotation = 0.0f;
	camera->setPosition(60.0f, 15.0f, 60.0f);
	constantFactor = 0.001f;
	linearFactor = 0.001f;
	quadraticFactor = 0.001f;
	minTessFactor = 4.0f;
	maxTessFactor = 64.0f;
	sobelScaleFactor = 1.0f;
	normalsToggle = false;
	motionBlurToggle = true;
	motionBlurXFactor = 5.0f;
	motionBlurYFactor = 5.0f;
	bloomToggle = true;
	glowIntensities[0] = 1.05f;
	glowIntensities[1] = 1.05f;
	glowIntensities[2] = 1.05f;
}

App1::~App1()
{
	//Tidy up all pointers
	BaseApplication::~BaseApplication();

	if (skybox)
	{
		delete skybox;
		skybox = NULL;
	}

	if (terrainPlane)
	{
		delete terrainPlane;
		terrainPlane = NULL;
	}

	if (waterPlane)
	{
		delete waterPlane;
		waterPlane = NULL;
	}

	if (textureShader)
	{
		delete textureShader;
		textureShader = NULL;
	}

	if (waterShader)
	{
		delete waterShader;
		waterShader = NULL;
	}

	if (terrainTessellationShader)
	{
		delete terrainTessellationShader;
		terrainTessellationShader = NULL;
	}

	if (terrainDepthShader)
	{
		delete terrainDepthShader;
		terrainDepthShader = NULL;
	}

	if (depthShader)
	{
		delete depthShader;
		depthShader = NULL;
	}

	if (shadowShader)
	{
		delete shadowShader;
		shadowShader = NULL;
	}

	if (sunLight)
	{
		delete sunLight;
		sunLight = NULL;
	}

	if (spotLight)
	{
		delete spotLight;
		spotLight = NULL;
	}

	if (hillPointLight)
	{
		delete hillPointLight;
		hillPointLight = NULL;
	}

	if (billboardGeometryShader)
	{
		delete billboardGeometryShader;
		billboardGeometryShader = NULL;
	}

	if (glowShader)
	{
		delete glowShader;
		glowShader = NULL;
	}

	if (gaussianShader)
	{
		delete gaussianShader;
		gaussianShader = NULL;
	}

	if (blendShader)
	{
		delete blendShader;
		blendShader = NULL;
	}

	if (sunShadowMap)
	{
		delete sunShadowMap;
		sunShadowMap = NULL;
	}

	if (spotShadowMap)
	{
		delete spotShadowMap;
		spotShadowMap = NULL;
	}
}

void App1::InitializeShaders(HWND hwnd, int screenWidth, int screenHeight)
{
	//Create all shader objects
	textureShader = new TextureShader(renderer->getDevice(), hwnd);
	clipPlaneShader = new ClipPlaneShader(renderer->getDevice(), hwnd);
	waterShader = new WaterShader(renderer->getDevice(), hwnd);
	terrainTessellationShader = new TerrainTessellationShader(renderer->getDevice(), hwnd);
	depthShader = new DepthShader(renderer->getDevice(), hwnd);
	terrainDepthShader = new TerrainDepthShader(renderer->getDevice(), hwnd);
	shadowShader = new ShadowShader(renderer->getDevice(), hwnd);
	billboardGeometryShader = new BillboardGeometryShader(renderer->getDevice(), hwnd);
	motionBlurShader = new MotionBlurShader(renderer->getDevice(), hwnd);
	glowShader = new GlowShader(renderer->getDevice(), hwnd);
	gaussianShader = new GaussianBlurShader(renderer->getDevice(), hwnd);
	blendShader = new BlendShader(renderer->getDevice(), hwnd);
}

void App1::InitializeLighting()
{
	//Create all lights, set all default values for properties
	sunLight = new Light;
	sunLightAmbient[0] = 0.2f; sunLightAmbient[1] = 0.2f; sunLightAmbient[2] = 0.2f;
	sunLightDiffuse[0] = 0.5f; sunLightDiffuse[1] = 0.5f; sunLightDiffuse[2] = 0.5f;
	sunLightPosition[0] = 50.0f; sunLightPosition[1] = 40.0f; sunLightPosition[2] = 70.0f;
	sunLightDirection[0] = 1.0f; sunLightDirection[1] = -0.75f; sunLightDirection[2] = 1.0f;

	spotLight = new Light;
	spotLightDiffuse[0] = 0.0f; spotLightDiffuse[1] = 1.0f; spotLightDiffuse[2] = 0.0f;
	spotLightPosition[0] = -50.0f; spotLightPosition[1] = 20.0f; spotLightPosition[2] = 78.0f;
	spotLightDirection[0] = 0.0f; spotLightDirection[1] = -1.0f; spotLightDirection[2] = 0.0f;

	hillPointLight = new Light;
	hillPointLightDiffuse[0] = 1.0f; hillPointLightDiffuse[1] = 0.0f; hillPointLightDiffuse[2] = 0.0f;
	hillPointLightPosition[0] = 136.0f; hillPointLightPosition[1] = 0.0f; hillPointLightPosition[2] = 96.0f;
}

void App1::SetLights()
{
	//Update sun light properties
	sunLight->setAmbientColour(sunLightAmbient[0], sunLightAmbient[1], sunLightAmbient[2], 1.0f);
	sunLight->setDiffuseColour(sunLightDiffuse[0], sunLightDiffuse[1], sunLightDiffuse[2], 1.0f);
	sunLight->setDirection(sunLightDirection[0], sunLightDirection[1], sunLightDirection[2]);
	sunLight->setPosition(sunLightPosition[0], sunLightPosition[1], sunLightPosition[2]);
	sunLight->generateOrthoMatrix(350.0f, 350.0f, 0.1f, 200.0f);

	//Update spot light properties
	spotLight->setDiffuseColour(spotLightDiffuse[0], spotLightDiffuse[1], spotLightDiffuse[2], 1.0f);
	spotLight->setPosition(spotLightPosition[0], spotLightPosition[1], spotLightPosition[2]);
	spotLight->setDirection(spotLightDirection[0], spotLightDirection[1], spotLightDirection[2]);
	spotLight->generateProjectionMatrix(0.1f, 200.0f);

	//Update point light properties
	hillPointLight->setDiffuseColour(hillPointLightDiffuse[0], hillPointLightDiffuse[1], hillPointLightDiffuse[2], 1.0f);
	hillPointLight->setPosition(hillPointLightPosition[0], hillPointLightPosition[1], hillPointLightPosition[2]);
	hillPointLight->generateProjectionMatrix(0.1f, 200.0f);

	//Update the shaders's reference to the sun light
	shadowShader->SetSunLight(sunLight);
	waterShader->SetSunLight(sunLight);
	terrainTessellationShader->SetSunLight(sunLight);
	billboardGeometryShader->SetSunLight(sunLight);

	//Update the shaders's reference to the point light
	shadowShader->SetHillLight(hillPointLight);
	waterShader->SetHillLight(hillPointLight);
	terrainTessellationShader->SetHillLight(hillPointLight);
	billboardGeometryShader->SetHillLight(hillPointLight);

	//Update the shaders's reference to the spot light
	shadowShader->SetSpotLight(spotLight);
	waterShader->SetSpotLight(spotLight);
	terrainTessellationShader->SetSpotLight(spotLight);
	billboardGeometryShader->SetSpotLight(spotLight);
}

void App1::SetAlphaToCoverageEnabled(bool enabled)
{
	//Create the descriptions
	ID3D11BlendState* blendState;
	D3D11_BLEND_DESC blendDesc;

	float blendFactor[4] = { 0.0f, 0.0f, 0.0f, 0.0f };
	UINT sampleMask = 0xffffffff;

	//Set the default values as per MSDN
	blendDesc.AlphaToCoverageEnable = enabled;
	blendDesc.IndependentBlendEnable = false;
	blendDesc.RenderTarget[0].BlendEnable = true;
	blendDesc.RenderTarget[0].SrcBlend = D3D11_BLEND_ONE;
	blendDesc.RenderTarget[0].DestBlend = D3D11_BLEND_ZERO;
	blendDesc.RenderTarget[0].BlendOp = D3D11_BLEND_OP_ADD;
	blendDesc.RenderTarget[0].SrcBlendAlpha = D3D11_BLEND_ONE;
	blendDesc.RenderTarget[0].DestBlendAlpha = D3D11_BLEND_ZERO;
	blendDesc.RenderTarget[0].BlendOpAlpha = D3D11_BLEND_OP_ADD;
	blendDesc.RenderTarget[0].RenderTargetWriteMask = D3D11_COLOR_WRITE_ENABLE_ALL;

	//Set the blend state of the renderer and release
	renderer->getDevice()->CreateBlendState(&blendDesc, &blendState);
	renderer->getDeviceContext()->OMSetBlendState(blendState, blendFactor, sampleMask);
	blendState->Release();
	blendState = 0;
}

void App1::GetGeometryPositions()
{
	//Create and push back all the meshes
	geometryMeshes.push_back({ Grass, new BillboardMesh(renderer->getDevice(), renderer->getDeviceContext()), textureMgr->getTexture(L"billboardGrass"), 44.0f, -0.75f, 81.0f });
	geometryMeshes.push_back({ Grass, new BillboardMesh(renderer->getDevice(), renderer->getDeviceContext()), textureMgr->getTexture(L"billboardGrass"), 44.5f, -1.0f, 79.5f });
	geometryMeshes.push_back({ Grass, new BillboardMesh(renderer->getDevice(), renderer->getDeviceContext()), textureMgr->getTexture(L"billboardGrass"), 43.0f, -1.0f, 81.0f });
	geometryMeshes.push_back({ Grass, new BillboardMesh(renderer->getDevice(), renderer->getDeviceContext()), textureMgr->getTexture(L"billboardGrass"), 43.5f, 0.0f, 79.0f });
	geometryMeshes.push_back({ Grass, new BillboardMesh(renderer->getDevice(), renderer->getDeviceContext()), textureMgr->getTexture(L"billboardGrass"), 44.0f, -1.0f, 79.5f });
	geometryMeshes.push_back({ Grass, new BillboardMesh(renderer->getDevice(), renderer->getDeviceContext()), textureMgr->getTexture(L"billboardGrass"), 44.0f, -0.5f, 79.0f });
	geometryMeshes.push_back({ Grass, new BillboardMesh(renderer->getDevice(), renderer->getDeviceContext()), textureMgr->getTexture(L"billboardGrass"), 44.0f, -0.5f, 80.0f });
	geometryMeshes.push_back({ Grass, new BillboardMesh(renderer->getDevice(), renderer->getDeviceContext()), textureMgr->getTexture(L"billboardGrass"), 45.0f, -1.0f, 80.0f });
	geometryMeshes.push_back({ Tree, new BillboardMesh(renderer->getDevice(), renderer->getDeviceContext()), textureMgr->getTexture(L"billboardTree"), 105.0f, 2.5f, 123.0f });
	geometryMeshes.push_back({ Tree, new BillboardMesh(renderer->getDevice(), renderer->getDeviceContext()), textureMgr->getTexture(L"billboardTree"), 125.0f, 3.0f, 128.0f });
}

bool App1::frame()
{
	bool result;

	result = BaseApplication::frame();
	if (!result)
	{
		return false;
	}

	result = render();
	if (!result)
	{
		return false;
	}

	return true;
}

bool App1::render()
{
	waterRotation += 0.5f * timer->getTime();	//Increment the water

	renderer->beginScene(0.39f, 0.58f, 0.92f, 1.0f);

	camera->update();	//Update the camera

	SetLights();	//Update the lights

	//Perform all the passes
	depthPass();
	reflectionPass();
	refractionPass();
	scenePass();
	bloomPass();
	motionBlurPass();
	finalPass();

	gui();	//Render the UI

	renderer->endScene();
	return true;
}

void App1::depthPass()
{
	#pragma region Sun Shadow
	sunShadowMap->BindDsvAndSetNullRenderTarget(renderer->getDeviceContext());

	sunLight->generateViewMatrix();
	XMMATRIX lightViewMatrix = sunLight->getViewMatrix();
	XMMATRIX lightProjectionMatrix = sunLight->getOrthoMatrix();	//Directional Light - Use Ortho
	XMMATRIX worldMatrix = renderer->getWorldMatrix();

	terrainPlane->sendData(renderer->getDeviceContext());
	terrainDepthShader->setShaderParameters(renderer->getDeviceContext(), worldMatrix, lightViewMatrix, lightProjectionMatrix, camera->getPosition(), minTessFactor, maxTessFactor, textureMgr->getTexture(L"testDisplacement"));
	terrainDepthShader->render(renderer->getDeviceContext(), terrainPlane->getIndexCount());

	hutModel->sendData(renderer->getDeviceContext());
	depthShader->setShaderParameters(renderer->getDeviceContext(), XMMatrixScaling(0.45f, 0.45f, 0.45f) * XMMatrixRotationRollPitchYaw(1.5708f, 0.5f, 0.0f) * XMMatrixTranslation(70.0f, 2.5f, 96.0f), lightViewMatrix, lightProjectionMatrix);
	depthShader->render(renderer->getDeviceContext(), hutModel->getIndexCount());

	renderer->setBackBufferRenderTarget();
	renderer->resetViewport();
	#pragma endregion

	#pragma region Spot Shadow
	spotShadowMap->BindDsvAndSetNullRenderTarget(renderer->getDeviceContext());

	//Get the correct view matrix for the direction
	if (spotLight->getDirection().x == 0.0f && spotLight->getDirection().z == 0.0f)
	{
		lightViewMatrix = GetYAxisViewMatrix();

		if (spotLight->getDirection().y < 0.0f)
		{
			lightViewMatrix = -lightViewMatrix;
		}
	}

	else
	{
		spotLight->generateViewMatrix();
		lightViewMatrix = spotLight->getViewMatrix();
	}

	lightProjectionMatrix = spotLight->getProjectionMatrix();	//Spot - Use Projection
	worldMatrix = renderer->getWorldMatrix();

	terrainPlane->sendData(renderer->getDeviceContext());
	terrainDepthShader->setShaderParameters(renderer->getDeviceContext(), worldMatrix, lightViewMatrix, lightProjectionMatrix, camera->getPosition(), minTessFactor, maxTessFactor, textureMgr->getTexture(L"testDisplacement"));
	terrainDepthShader->render(renderer->getDeviceContext(), terrainPlane->getIndexCount());

	hutModel->sendData(renderer->getDeviceContext());
	depthShader->setShaderParameters(renderer->getDeviceContext(), XMMatrixScaling(0.45f, 0.45f, 0.45f) * XMMatrixRotationRollPitchYaw(1.5708f, 0.5f, 0.0f) * XMMatrixTranslation(70.0f, 2.5f, 96.0f), lightViewMatrix, lightProjectionMatrix);
	depthShader->render(renderer->getDeviceContext(), hutModel->getIndexCount());

	renderer->setBackBufferRenderTarget();
	renderer->resetViewport();
	#pragma endregion

	#pragma region Point Shadow
	//All directions to sample
	XMFLOAT3 lightDirections[6] =
	{
		XMFLOAT3(0.0f, 1.0f, 0.0f),
		XMFLOAT3(0.0f, -1.0f, 0.0f),
		XMFLOAT3(1.0f, 0.0f, 0.0f),
		XMFLOAT3(-1.0f, 0.0f, 0.0f),
		XMFLOAT3(0.0f, 0.0f, 1.0f),
		XMFLOAT3(0.0f, 0.0f, -1.0f)
	};

	for (int i = 0; i < 6; i++)
	{
		//Loop through all directions, set each one and sample
		hillPointLight->setDirection(lightDirections[i].x, lightDirections[i].y, lightDirections[i].z);
		hillShadowMaps[i]->BindDsvAndSetNullRenderTarget(renderer->getDeviceContext());

		if (hillPointLight->getDirection().x == 0.0f && hillPointLight->getDirection().z == 0.0f)
		{
			lightViewMatrix = GetYAxisViewMatrix();

			if (hillPointLight->getDirection().y < 0.0f)
			{
				lightViewMatrix = -lightViewMatrix;
			}
		}

		else
		{
			hillPointLight->generateViewMatrix();
			lightViewMatrix = hillPointLight->getViewMatrix();
		}

		lightProjectionMatrix = hillPointLight->getProjectionMatrix();	//Point - Use Projection
		worldMatrix = renderer->getWorldMatrix();

		terrainPlane->sendData(renderer->getDeviceContext());
		terrainDepthShader->setShaderParameters(renderer->getDeviceContext(), worldMatrix, lightViewMatrix, lightProjectionMatrix, camera->getPosition(), minTessFactor, maxTessFactor, textureMgr->getTexture(L"testDisplacement"));
		terrainDepthShader->render(renderer->getDeviceContext(), terrainPlane->getIndexCount());

		hutModel->sendData(renderer->getDeviceContext());
		depthShader->setShaderParameters(renderer->getDeviceContext(), XMMatrixScaling(0.45f, 0.45f, 0.45f) * XMMatrixRotationRollPitchYaw(1.5708f, 0.5f, 0.0f) * XMMatrixTranslation(70.0f, 2.5f, 96.0f), lightViewMatrix, lightProjectionMatrix);
		depthShader->render(renderer->getDeviceContext(), hutModel->getIndexCount());

		renderer->setBackBufferRenderTarget();
		renderer->resetViewport();
	}
	#pragma endregion
}

void App1::reflectionPass()
{
	//Anything above the water, camera below
	//Render to the reflection texture
	reflectionRenderTexture->setRenderTarget(renderer->getDeviceContext());
	reflectionRenderTexture->clearRenderTarget(renderer->getDeviceContext(), 0.39f, 0.58f, 0.92f, 1.0f);

	XMMATRIX viewMatrix = GetReflectionViewMatrix();
	XMMATRIX worldMatrix = renderer->getWorldMatrix();
	XMMATRIX projectionMatrix = renderer->getProjectionMatrix();

	//Render the skybox
	renderer->setZBuffer(false);
	skybox->sendData(renderer->getDeviceContext());

	//We rotate the skybox around so it reflects correctly, then position it so the camera is in the centre
	textureShader->setShaderParameters(renderer->getDeviceContext(),XMMatrixRotationRollPitchYaw(3.14159f, 3.14159f, 0.0f) * XMMatrixTranslation(camera->getPosition().x + 0.5f, camera->getPosition().y + 0.5f, camera->getPosition().z + 0.5f), camera->getViewMatrix(), projectionMatrix, textureMgr->getTexture(L"skybox"));
	textureShader->render(renderer->getDeviceContext(), skybox->getIndexCount());
	renderer->setZBuffer(true);

	//Render the cube, using the clip plane shader to clip it if needed
	refractionWoodCube->sendData(renderer->getDeviceContext());
	clipPlaneShader->setShaderParameters(renderer->getDeviceContext(), XMMatrixRotationRollPitchYaw(0.0f, waterRotation, 0.0f) * XMMatrixTranslation(65.0f, -5.0f, 60.0f), viewMatrix, projectionMatrix, textureMgr->getTexture(L"wood"), XMFLOAT4(0.0f, 1.0f, 0.0f, 0.0f));
	clipPlaneShader->render(renderer->getDeviceContext(), refractionWoodCube->getIndexCount());

	renderer->setBackBufferRenderTarget();
}

void App1::refractionPass()
{
	//Anything under the water, camera above
	//Render to the refraction texture
	refractionRenderTexture->setRenderTarget(renderer->getDeviceContext());
	refractionRenderTexture->clearRenderTarget(renderer->getDeviceContext(), 0.39f, 0.58f, 0.92f, 1.0f);

	XMMATRIX worldMatrix = renderer->getWorldMatrix();
	XMMATRIX viewMatrix = camera->getViewMatrix();
	XMMATRIX projectionMatrix = renderer->getProjectionMatrix();

	//Render the skybox
	renderer->setZBuffer(false);
	skybox->SetPosition(XMFLOAT3(camera->getPosition().x - 0.5f, camera->getPosition().y - 0.5f, camera->getPosition().z + 0.5f));
	skybox->sendData(renderer->getDeviceContext());
	textureShader->setShaderParameters(renderer->getDeviceContext(), skybox->GetWorldMatrix(), viewMatrix, projectionMatrix, textureMgr->getTexture(L"skybox"));
	textureShader->render(renderer->getDeviceContext(), skybox->getIndexCount());
	renderer->setZBuffer(true);

	//Render the cube, using the clip plane shader to clip it if needed
	refractionWoodCube->sendData(renderer->getDeviceContext());
	clipPlaneShader->setShaderParameters(renderer->getDeviceContext(), XMMatrixRotationRollPitchYaw(0.0f, waterRotation, 0.0f) * XMMatrixTranslation(65.0f, -5.0f, 60.0f), viewMatrix, projectionMatrix, textureMgr->getTexture(L"wood"), XMFLOAT4(0.0f, -1.0f, 0.0f, 0.1f));
	clipPlaneShader->render(renderer->getDeviceContext(), refractionWoodCube->getIndexCount());

	renderer->setBackBufferRenderTarget();
}

void App1::scenePass()
{
	//Loop through and create SRVs from the point light shadow maps
	ID3D11ShaderResourceView* hillShadowResources[6];
	for (int i = 0; i < 6; i++)
	{
		hillShadowResources[i] = hillShadowMaps[i]->getDepthMapSRV();
	}

	XMMATRIX worldMatrix = renderer->getWorldMatrix();
	XMMATRIX viewMatrix = camera->getViewMatrix();
	XMMATRIX projectionMatrix = renderer->getProjectionMatrix();

	//Render to the scene texture
	sceneRenderTexture->setRenderTarget(renderer->getDeviceContext());
	sceneRenderTexture->clearRenderTarget(renderer->getDeviceContext(), 0.39f, 0.58f, 0.92f, 1.0f);

	//Render the skybox
	renderer->setZBuffer(false);
	skybox->SetPosition(XMFLOAT3(camera->getPosition().x - 0.5f, camera->getPosition().y - 0.5f, camera->getPosition().z + 0.5f));
	skybox->sendData(renderer->getDeviceContext());
	textureShader->setShaderParameters(renderer->getDeviceContext(), skybox->GetWorldMatrix(), viewMatrix, projectionMatrix, textureMgr->getTexture(L"skybox"));
	textureShader->render(renderer->getDeviceContext(), skybox->getIndexCount());
	renderer->setZBuffer(true);

	//About to render the geometry, set the correct alpha state
	renderer->setAlphaBlending(true);
	SetAlphaToCoverageEnabled(true);

	//Loop through and render all geometry
	for (auto &theMesh : geometryMeshes)
	{
		XMMATRIX meshWorld = XMMatrixTranslation(theMesh.xPosition, theMesh.yPosition, theMesh.zPosition);
		theMesh.mesh->sendData(renderer->getDeviceContext());
		billboardGeometryShader->setShaderParameters(renderer->getDeviceContext(), meshWorld, viewMatrix, projectionMatrix, camera->getPosition(), theMesh.type, theMesh.texture, sunShadowMap->getDepthMapSRV(), spotShadowMap->getDepthMapSRV(), hillShadowResources, constantFactor / 10.0f, linearFactor / 100.0f, quadraticFactor / 1000.0f);
		billboardGeometryShader->render(renderer->getDeviceContext(), theMesh.mesh->getIndexCount());
	}

	//All geometry rendered, reset the alpha state
	renderer->setAlphaBlending(false);
	SetAlphaToCoverageEnabled(false);

	//Render the cube
	refractionWoodCube->sendData(renderer->getDeviceContext());
	textureShader->setShaderParameters(renderer->getDeviceContext(), XMMatrixRotationRollPitchYaw(0.0f, waterRotation, 0.0f) * XMMatrixTranslation(65.0f, -5.0f, 60.0f), viewMatrix, projectionMatrix, textureMgr->getTexture(L"wood"));
	textureShader->render(renderer->getDeviceContext(), refractionWoodCube->getIndexCount());

	//Render the main terrain, tessellating, displacing and shadowing it
	terrainPlane->sendData(renderer->getDeviceContext());
	terrainTessellationShader->setShaderParameters(renderer->getDeviceContext(), worldMatrix, viewMatrix, projectionMatrix, camera->getPosition(), minTessFactor, maxTessFactor, sobelScaleFactor, textureMgr->getTexture(L"testDisplacement"), textureMgr->getTexture(L"textureMap"), textureMgr->getTexture(L"grass"), textureMgr->getTexture(L"rock"), textureMgr->getTexture(L"dirt"), textureMgr->getTexture(L"sand"), sunShadowMap->getDepthMapSRV(), spotShadowMap->getDepthMapSRV(), hillShadowResources, constantFactor / 10.0f, linearFactor / 100.0f, quadraticFactor / 1000.0f, normalsToggle);
	terrainTessellationShader->render(renderer->getDeviceContext(), terrainPlane->getIndexCount());

	//Render the water plane, passing the reflection, refraction, DUDV and normal textures
	waterPlane->sendData(renderer->getDeviceContext());
	waterShader->setShaderParameters(renderer->getDeviceContext(), XMMatrixScaling(2.0f, 2.0f, 2.0f) * XMMatrixTranslation(0.0f, -0.1f, 0.0f) * worldMatrix, viewMatrix, projectionMatrix, GetReflectionViewMatrix(), reflectionRenderTexture->getShaderResourceView(), refractionRenderTexture->getShaderResourceView(), textureMgr->getTexture(L"dudv"), textureMgr->getTexture(L"waterNormal"), camera->getPosition(), timer->getTime());
	waterShader->render(renderer->getDeviceContext(), waterPlane->getIndexCount());

	//Render the hut, shadowing it
	hutModel->sendData(renderer->getDeviceContext());
	shadowShader->setShaderParameters(renderer->getDeviceContext(), XMMatrixScaling(0.45f, 0.45f, 0.45f) * XMMatrixRotationRollPitchYaw(1.5708f, 0.5f, 0.0f) * XMMatrixTranslation(70.0f, 2.5f, 96.0f), viewMatrix, projectionMatrix, textureMgr->getTexture(L"hutTexture"), sunShadowMap->getDepthMapSRV(), spotShadowMap->getDepthMapSRV(), hillShadowResources, constantFactor / 10.0f, linearFactor / 100.0f, quadraticFactor / 1000.0f);
	shadowShader->render(renderer->getDeviceContext(), hutModel->getIndexCount());

	renderer->setBackBufferRenderTarget();
	renderer->resetViewport();
}

void App1::bloomPass()
{
	if (bloomToggle)	//If we should apply bloom
	{
		//Render to the glow texture
		glowTexture->setRenderTarget(renderer->getDeviceContext());
		glowTexture->clearRenderTarget(renderer->getDeviceContext(), 0.39f, 0.58f, 0.92f, 1.0f);

		XMMATRIX worldMatrix = renderer->getWorldMatrix();
		XMMATRIX orthoViewMatrix = camera->getOrthoViewMatrix();
		XMMATRIX orthoMatrix = renderer->getOrthoMatrix();

		//Apply the glow intensities and effect onto the scene texture and save it into the glow texture
		sceneOrtho->sendData(renderer->getDeviceContext());
		glowShader->setShaderParameters(renderer->getDeviceContext(), worldMatrix, orthoViewMatrix, orthoMatrix, sceneRenderTexture->getShaderResourceView(), XMFLOAT4(glowIntensities[0], glowIntensities[1], glowIntensities[2], 0.0f));
		glowShader->render(renderer->getDeviceContext(), sceneOrtho->getIndexCount());

		renderer->setBackBufferRenderTarget();
		renderer->resetViewport();

		//Render to the glow blur texture
		glowBlurTexture->setRenderTarget(renderer->getDeviceContext());
		glowBlurTexture->clearRenderTarget(renderer->getDeviceContext(), 0.39f, 0.58f, 0.92f, 1.0f);

		//Apply a gaussian blur to the glowed scene texture
		sceneOrtho->sendData(renderer->getDeviceContext());
		gaussianShader->setShaderParameters(renderer->getDeviceContext(), worldMatrix, orthoViewMatrix, orthoMatrix, glowTexture->getShaderResourceView());
		gaussianShader->render(renderer->getDeviceContext(), sceneOrtho->getIndexCount());

		renderer->setBackBufferRenderTarget();
		renderer->resetViewport();

		//Render to the final bloom texture
		finalBloomTexture->setRenderTarget(renderer->getDeviceContext());
		finalBloomTexture->clearRenderTarget(renderer->getDeviceContext(), 0.39f, 0.58f, 0.92f, 1.0f);

		//Blend the scene texture with the glowed blurred texture and save it into the final bloom texture
		sceneOrtho->sendData(renderer->getDeviceContext());
		blendShader->setShaderParameters(renderer->getDeviceContext(), worldMatrix, orthoViewMatrix, orthoMatrix, sceneRenderTexture->getShaderResourceView(), glowBlurTexture->getShaderResourceView());
		blendShader->render(renderer->getDeviceContext(), sceneOrtho->getIndexCount());

		renderer->setBackBufferRenderTarget();
		renderer->resetViewport();

		//Render to the scene texture
		sceneRenderTexture->setRenderTarget(renderer->getDeviceContext());
		sceneRenderTexture->clearRenderTarget(renderer->getDeviceContext(), 0.39f, 0.58f, 0.92f, 1.0f);

		renderToSceneOrtho(finalBloomTexture);	//Render the final bloom texture to the scene texture

		renderer->setBackBufferRenderTarget();
		renderer->resetViewport();
	}
}

void App1::motionBlurPass()
{
	if (motionBlurToggle)	//If we should apply motion blur
	{
		//Render to the motion blur texture
		motionBlurTexture->setRenderTarget(renderer->getDeviceContext());
		motionBlurTexture->clearRenderTarget(renderer->getDeviceContext(), 0.39f, 0.58f, 0.92f, 1.0f);

		XMMATRIX worldMatrix = renderer->getWorldMatrix();
		XMMATRIX orthoViewMatrix = camera->getOrthoViewMatrix();
		XMMATRIX orthoMatrix = renderer->getOrthoMatrix();
		
		//Calculate the right, up and forward vectors
		XMMATRIX camRotationMatrix = XMMatrixRotationRollPitchYaw(camera->getRotation().x * 0.0174532f, camera->getRotation().y * 0.0174532f, 0);
		XMVECTOR camTarget = XMVector3TransformCoord(XMVectorSet(0.0f, 0.0f, 1.0f, 0.0f), camRotationMatrix);
		camTarget = XMVector3Normalize(camTarget);

		XMMATRIX RotateYTempMatrix;
		RotateYTempMatrix = XMMatrixRotationY(camera->getRotation().y * 0.0174532f);

		XMVECTOR camRight = XMVector3TransformCoord(XMVectorSet(1.0f, 0.0f, 0.0f, 0.0f), RotateYTempMatrix);
		XMVECTOR camUp = XMVector3TransformCoord(XMVectorSet(0.0f, 1.0f, 0.0f, 0.0f), RotateYTempMatrix);
		XMVECTOR camForward = XMVector3TransformCoord(XMVectorSet(0.0f, 0.0f, 1.0f, 0.0f), RotateYTempMatrix);

		//The motion vector is the difference in positions
		XMVECTOR motionVector = XMVectorSet(camera->getPosition().x - previousCameraPosition.x, camera->getPosition().y - previousCameraPosition.y, camera->getPosition().z - previousCameraPosition.z, 0.0f);

		XMVECTOR xMotion = XMVector3Dot(motionVector, camRight);	//X motion is the dot product of our motion vs the right vector
		XMVECTOR yMotion = XMVector3Dot(motionVector, camForward);	//Y motion is the dot product of our motion vs the forward vector
		float xDirection = (*xMotion.m128_f32) * motionBlurXFactor;	//Calculate X direction (every component of the vector will have the dot product value)
		float yDirection = (*yMotion.m128_f32) * motionBlurYFactor;	//Calculate Y direction (every component of the vector will have the dot product value)

		//Apply motion blur onto the scene render texture and save it to the motion blur texture
		sceneOrtho->sendData(renderer->getDeviceContext());
		motionBlurShader->setShaderParameters(renderer->getDeviceContext(), worldMatrix, orthoViewMatrix, orthoMatrix, sceneRenderTexture->getShaderResourceView(), XMFLOAT2(xDirection, yDirection));
		motionBlurShader->render(renderer->getDeviceContext(), sceneOrtho->getIndexCount());

		//Render to the scene texture
		sceneRenderTexture->setRenderTarget(renderer->getDeviceContext());
		sceneRenderTexture->clearRenderTarget(renderer->getDeviceContext(), 0.39f, 0.58f, 0.92f, 1.0f);

		renderToSceneOrtho(motionBlurTexture);	//Render the motion blurred texture to the scene render texture

		renderer->setBackBufferRenderTarget();
		renderer->resetViewport();
	}
}

void App1::finalPass()
{
	renderToSceneOrtho(sceneRenderTexture);	//Render the final scene texture
	previousCameraPosition = camera->getPosition();
}

void App1::renderToSceneOrtho(RenderTexture* texture)
{
	renderer->setZBuffer(false);

	XMMATRIX worldMatrix = renderer->getWorldMatrix();
	XMMATRIX orthoMatrix = renderer->getOrthoMatrix();
	XMMATRIX orthoViewMatrix = camera->getOrthoViewMatrix();

	//Render the passed in texture to the scene ortho mesh
	sceneOrtho->sendData(renderer->getDeviceContext());
	textureShader->setShaderParameters(renderer->getDeviceContext(), worldMatrix, orthoViewMatrix, orthoMatrix, texture->getShaderResourceView());
	textureShader->render(renderer->getDeviceContext(), sceneOrtho->getIndexCount());

	renderer->setZBuffer(true);
}

void App1::DebugShowReflectRefract()
{
	XMMATRIX worldMatrix = renderer->getWorldMatrix();
	XMMATRIX orthoViewMatrix = camera->getOrthoViewMatrix();	// Default camera position for orthographic rendering
	XMMATRIX orthoMatrix = renderer->getOrthoMatrix();  // ortho matrix for 2D rendering

	reflectionOrtho->sendData(renderer->getDeviceContext());
	textureShader->setShaderParameters(renderer->getDeviceContext(), worldMatrix, orthoViewMatrix, orthoMatrix, reflectionRenderTexture->getShaderResourceView());
	textureShader->render(renderer->getDeviceContext(), reflectionOrtho->getIndexCount());

	refractionOrtho->sendData(renderer->getDeviceContext());
	textureShader->setShaderParameters(renderer->getDeviceContext(), worldMatrix, orthoViewMatrix, orthoMatrix, refractionRenderTexture->getShaderResourceView());
	textureShader->render(renderer->getDeviceContext(), refractionOrtho->getIndexCount());
}

void App1::DebugShowShadowMap()
{
	XMMATRIX worldMatrix = renderer->getWorldMatrix();
	XMMATRIX orthoViewMatrix = camera->getOrthoViewMatrix();	// Default camera position for orthographic rendering
	XMMATRIX orthoMatrix = renderer->getOrthoMatrix();  // ortho matrix for 2D rendering

	shadowOrtho->sendData(renderer->getDeviceContext());
	textureShader->setShaderParameters(renderer->getDeviceContext(), worldMatrix, orthoViewMatrix, orthoMatrix, spotShadowMap->getDepthMapSRV());
	textureShader->render(renderer->getDeviceContext(), refractionOrtho->getIndexCount());
}

void App1::gui()
{
	// Force turn off unnecessary shader stages.
	renderer->getDeviceContext()->GSSetShader(NULL, NULL, 0);
	renderer->getDeviceContext()->HSSetShader(NULL, NULL, 0);
	renderer->getDeviceContext()->DSSetShader(NULL, NULL, 0);

	// Build UI
	ImGui::Text("FPS: %.2f", timer->getFPS());
	ImGui::Text("Camera X: %.2f", camera->getPosition().x);
	ImGui::Text("Camera Y: %.2f", camera->getPosition().y);
	ImGui::Text("Camera Z: %.2f", camera->getPosition().z);

	if (ImGui::CollapsingHeader("Light Options"))
	{
		if (ImGui::CollapsingHeader("Sun Light Options"))
		{
			ImGui::ColorEdit3("Sun Light Ambient", (float*)&sunLightAmbient);
			ImGui::ColorEdit3("Sun Light Diffuse", (float*)&sunLightDiffuse);
			ImGui::DragFloat3("Sun Light Direction", (float*)&sunLightDirection);
		}

		if (ImGui::CollapsingHeader("Hill Point Options"))
		{
			ImGui::ColorEdit3("Hill Light Diffuse", (float*)&hillPointLightDiffuse);
			ImGui::DragFloat3("Hill Light Position", (float*)&hillPointLightPosition);
		}

		if (ImGui::CollapsingHeader("Spot Options"))
		{
			ImGui::ColorEdit3("Spot Light Diffuse", (float*)&spotLightDiffuse);
			ImGui::DragFloat3("Spot Light Position", (float*)&spotLightPosition);
			ImGui::DragFloat3("Spot Light Direction", (float*)&spotLightDirection);

			if (ImGui::CollapsingHeader("Attenuation Options"))
			{
				ImGui::SliderFloat("Constant Factor", &constantFactor, 0.001f, 1.0f);
				ImGui::SliderFloat("Linear Factor", &linearFactor, 0.001f, 1.0f);
				ImGui::SliderFloat("Quadratic Factor", &quadraticFactor, 0.001f, 1.0f);
			}
		}
	}

	if (ImGui::CollapsingHeader("Tessellation Options"))
	{
		ImGui::SliderFloat("Minimum Factor", &minTessFactor, 4, 64);
		ImGui::SliderFloat("Maximum Factor", &maxTessFactor, 4, 64);
	}

	if (ImGui::CollapsingHeader("Post-Processing Options"))
	{
		if (ImGui::CollapsingHeader("Bloom Options"))
		{
			ImGui::Checkbox("Bloom Enabled", &bloomToggle);
			ImGui::SliderFloat("R Glow Intensity", &glowIntensities[0], 1, 2);
			ImGui::SliderFloat("G Glow Intensity", &glowIntensities[1], 1, 2);
			ImGui::SliderFloat("B Glow Intensity", &glowIntensities[2], 1, 2);
		}

		if (ImGui::CollapsingHeader("Motion Blur Options"))
		{
			ImGui::Checkbox("Motion Blur Enabled", &motionBlurToggle);
			ImGui::SliderFloat("X Factor", &motionBlurXFactor, 5, 50);
			ImGui::SliderFloat("Y Factor", &motionBlurYFactor, 5, 50);
		}
	}

	ImGui::SliderFloat("Sobel Scale Factor", &sobelScaleFactor, 0.5f, 2.0f);

	ImGui::Checkbox("Terrain Normals Mode", &normalsToggle);
	//ImGui::Checkbox("Wireframe mode", &wireframeToggle);

	// Render UI
	ImGui::Render();
	ImGui_ImplDX11_RenderDrawData(ImGui::GetDrawData());
}

XMMATRIX App1::GetReflectionViewMatrix()
{
	camera->getViewMatrix();

	XMVECTOR up, positionv, lookAt;
	float yaw, pitch, roll;
	XMMATRIX rotationMatrix;

	// Setup the vectors
	up = XMVectorSet(0.0f, 1.0, 0.0, 1.0f);
	positionv = XMLoadFloat3(new XMFLOAT3(camera->getPosition().x, camera->getPosition().y - (2.0f * (camera->getPosition().y)), camera->getPosition().z));
	lookAt = XMVectorSet(0.0, 0.0, 1.0f, 1.0f);

	// Set the yaw (Y axis), pitch (X axis), and roll (Z axis) rotations in radians.
	pitch = -(camera->getRotation().x * 0.0174532f);
	yaw = camera->getRotation().y * 0.0174532f;
	roll = camera->getRotation().z * 0.0174532f;

	// Create the rotation matrix from the yaw, pitch, and roll values.
	rotationMatrix = XMMatrixRotationRollPitchYaw(pitch, yaw, roll);

	// Transform the lookAt and up vector by the rotation matrix so the view is correctly rotated at the origin.
	lookAt = XMVector3TransformCoord(lookAt, rotationMatrix);
	up = XMVector3TransformCoord(up, rotationMatrix);

	// Translate the rotated camera position to the location of the viewer.
	lookAt = positionv + lookAt;

	// Finally create the view matrix from the three updated vectors.
	return XMMatrixLookAtLH(positionv, lookAt, up);
}

XMMATRIX App1::GetYAxisViewMatrix()
{
	float yDirection = hillPointLight->getDirection().y;
	yDirection > 0.0f ? yDirection = 1.0f : yDirection = -1.0f;

	XMFLOAT3 lightDirection = XMFLOAT3(0.0f, yDirection, 0.0f);
	XMFLOAT3 lightUp = XMFLOAT3(0.0f, 0.0f, 1.0f);

	XMVECTOR lightPosVector = XMLoadFloat3(&hillPointLight->getPosition());
	XMVECTOR lightDirectionVector = XMLoadFloat3(&lightDirection);
	XMVECTOR lightUpVector = XMLoadFloat3(&lightUp);

	return XMMatrixLookToLH(lightPosVector, lightDirectionVector, lightUpVector);
}